from maya.api import OpenMaya
# from maya import OpenMayaMPx
import operator
import sys

maya_useNewAPI = True

class RRM3Node(OpenMaya.MPxNode):
    """
    A basic multiply node
    """
    TYPE_NAME = 'rrm3Node'
    TYPE_ID = OpenMaya.MTypeId(0xd802a0)

    OPERATOR = None
    INPUT_DEFAULT_VALUE = 1.0

    input1 = None
    input2 = None
    output = None

    def __init__(self):
        OpenMaya.MPxNode.__init__(self)
        self.connected_node = None

    @classmethod
    def creator(cls):
        # return OpenMayaMPx.asMPxPtr(cls())
        return cls()

    @classmethod
    def initialize(cls):
        pass

    def compute(self, plug, data_block):
        pass

    @classmethod
    def create_m_fn_attr_input(cls, input_int, suffix=''):
        m_fn_attr = OpenMaya.MFnNumericAttribute()
        k_float = OpenMaya.MFnNumericData.kFloat

        m_fn_attr_input = m_fn_attr.create(f'input{input_int}{suffix}', f'i{input_int}{suffix.lower()}',
                                           k_float, cls.INPUT_DEFAULT_VALUE)

        cls.set_attr_states(m_fn_attr=m_fn_attr, attr_type='input')

        # attach attribute to the node
        # print(f'{m_fn_attr_input=}', type(m_fn_attr_input))
        try:
            cls.addAttribute(m_fn_attr_input)
        except RuntimeError as err:
            sys.stderr.write(f"Failed to add attribute: {m_fn_attr_input} {err}")
            raise
        return m_fn_attr_input

    @classmethod
    def create_m_fn_attr_output(cls, suffix=''):
        m_fn_attr = OpenMaya.MFnNumericAttribute()
        k_float = OpenMaya.MFnNumericData.kFloat

        m_fn_attr_output = m_fn_attr.create(f'output{suffix}', f'out{suffix.lower()}', k_float)
        cls.set_attr_states(m_fn_attr=m_fn_attr, attr_type='output')

        # attach attribute to node
        cls.addAttribute(m_fn_attr_output)

        return m_fn_attr_output

    @classmethod
    def set_attr_states(cls, m_fn_attr, attr_type):
        if attr_type == 'input':
            readable, writable, keyable = False, True, True
        else:  # is output
            readable, writable, keyable = True, False, False

        m_fn_attr.readable = readable
        m_fn_attr.writable = writable
        m_fn_attr.keyable = keyable

    @classmethod
    def create_m_fn_compound(cls, long_name, short_name, children, readable, writable):
        """
        set up compound attribute (so output is connected to outputX,Y,Z)
        @param long_name:
        @param short_name:
        @param children:
        @param readable:
        @param writable:
        @return:
        """
        m_fn_compound = OpenMaya.MFnCompoundAttribute()

        m_fn_compound_object = m_fn_compound.create(long_name, short_name)
        m_fn_compound.readable = readable
        m_fn_compound.writable = writable

        for child in children:
            m_fn_compound.addChild(child)
        # attach attribute to node
        cls.addAttribute(m_fn_compound_object)

        return m_fn_compound_object

    def postConstructor(self):
        # This method is called when the node is created

        # keep the nodes around, even if there connections are deleted
        self.setExistWithoutInConnections(True)
        self.setExistWithoutOutConnections(True)

# floats ------------
class RRM3FloatNode(RRM3Node):
    """
    A basic float math parent class node
    """
    TYPE_NAME = 'rrm3Float'
    TYPE_ID = OpenMaya.MTypeId(0xd802a1)

    @classmethod
    def initialize(cls):
        # create a function set

        # create input attrs
        cls.input1 = cls.create_m_fn_attr_input(input_int=1)
        cls.input2 = cls.create_m_fn_attr_input(input_int=2)

        # create output attr
        cls.output = cls.create_m_fn_attr_output()

        # design circuitry
        # print(cls.input1)
        # print(cls.output)
        cls.attributeAffects(cls.input1, cls.output)
        cls.attributeAffects(cls.input2, cls.output)

    # @classmethod
    def compute(self, plug, data_block):
        if plug == self.output:
            input1_val = data_block.inputValue(self.input1).asFloat()
            input2_val = data_block.inputValue(self.input2).asFloat()
            # get output handle, set its new value, and set it clean.
            output_handle = data_block.outputValue(self.output)
            output_val =  self.OPERATOR(input1_val, input2_val)
            output_handle.setFloat(output_val)
            output_handle.setClean()


class RRM3FloatMultNode(RRM3FloatNode):
    """
    A basic multiply node
    """
    TYPE_NAME = 'rrm3FloatMult'
    TYPE_ID = OpenMaya.MTypeId(0xd802a2)
    OPERATOR = operator.mul


class RRM3FloatDivNode(RRM3FloatNode):
    """
    A basic divide node
    """
    TYPE_NAME = 'rrm3FloatDiv'
    TYPE_ID = OpenMaya.MTypeId(0xd802a3)
    OPERATOR = operator.truediv


class RRM3FloatAddNode(RRM3FloatNode):
    """
    A basic add node
    """
    TYPE_NAME = 'rrm3FloatAdd'
    TYPE_ID = OpenMaya.MTypeId(0xd802a4)

    OPERATOR = operator.add


class RRM3FloatSubtractNode(RRM3FloatNode):
    """
    A basic float subtract node
    """
    TYPE_NAME = 'rrm3FloatSubtract'
    TYPE_ID = OpenMaya.MTypeId(0xd802a5)

    OPERATOR = operator.sub


# ------ vectors
class RRM3Vec3Node(RRM3Node):
    """
    A vector3 math base class for a node
    """
    TYPE_NAME = 'rrm3Vec3'
    TYPE_ID = OpenMaya.MTypeId(0xd802b0)

    input1x = None
    input1y = None
    input1z = None
    input2x = None
    input2y = None
    input2z = None

    outputx = None
    outputy = None
    outputz = None

    @classmethod
    def initialize(cls):
        # inputs
        cls.input1x = cls.create_m_fn_attr_input(input_int=1, suffix='X')
        cls.input1y = cls.create_m_fn_attr_input(input_int=1, suffix='Y')
        cls.input1z = cls.create_m_fn_attr_input(input_int=1, suffix='Z')
        cls.input2x = cls.create_m_fn_attr_input(input_int=2, suffix='X')
        cls.input2y = cls.create_m_fn_attr_input(input_int=2, suffix='Y')
        cls.input2z = cls.create_m_fn_attr_input(input_int=2, suffix='Z')
        # output
        cls.outputx = cls.create_m_fn_attr_output(suffix='X')
        cls.outputy = cls.create_m_fn_attr_output(suffix='Y')
        cls.outputz = cls.create_m_fn_attr_output(suffix='Z')

        # create compounds
        cls.input1 = cls.create_m_fn_compound(long_name='input1', short_name='in1',
                                              children=[cls.input1x, cls.input1y, cls.input1z],
                                              readable=False, writable=True)

        cls.input2 = cls.create_m_fn_compound(long_name='input2', short_name='in2',
                                              children=[cls.input2x, cls.input2y, cls.input2z],
                                              readable=False, writable=True)

        cls.output = cls.create_m_fn_compound(long_name='output', short_name='out',
                                              children=[cls.outputx, cls.outputy, cls.outputz],
                                              readable=True, writable=False)

        # design circuitry
        cls.attributeAffects(cls.input1, cls.output)
        cls.attributeAffects(cls.input2, cls.output)

    def compute_vec3(self, data_block):

        input1_val = data_block.inputValue(self.input1).asFloat3()
        input2_val = data_block.inputValue(self.input2).asFloat3()
        # get output handle, set its new value, and set it clean.
        output_handle = data_block.outputValue(self.output)

        output_val = (self.OPERATOR(input1_val[0], input2_val[0]),
                      self.OPERATOR(input1_val[1], input2_val[1]),
                      self.OPERATOR(input1_val[2], input2_val[2]))
        output_handle.set3Float(output_val[0], output_val[1], output_val[2])
        output_handle.setClean()

    def get_float_input_output(self, plug):
        input1 = None
        input2 = None
        if plug == self.outputx:
            input1 = self.input1x
            input2 = self.input2x
        elif plug == self.outputy:
            input1 = self.input1y
            input2 = self.input2y
        elif plug == self.outputx:
            input1 = self.input1z
            input2 = self.input2z

        return input1, input2


    def compute(self, plug, data_block):
        if plug == self.output:
            self.compute_vec3(data_block=data_block)

        else:  # plug might be x, y, or z
            self.input1, self.input2 = self.get_float_input_output(plug=plug)
            if not self.input1 or not self.input2:
                return

            # use the float compute if it is not a vector
            input1_val = data_block.inputValue(self.input1).asFloat()
            input2_val = data_block.inputValue(self.input2).asFloat()
            # get output handle, set its new value, and set it clean.
            output_handle = data_block.outputValue(self.output)
            output_val =  self.OPERATOR(input1_val, input2_val)
            output_handle.setFloat(output_val)
            output_handle.setClean()

class RRM3Vec3MultNode(RRM3Vec3Node):
    """
    A vector3 multiply node
    """
    TYPE_NAME = 'rrm3Vec3Mult'
    TYPE_ID = OpenMaya.MTypeId(0xd802b1)
    OPERATOR = operator.mul

class RRM3Vec3DivNode(RRM3Vec3Node):
    """
    A vector3 divide node
    """
    TYPE_NAME = 'rrm3Vec3Div'
    TYPE_ID = OpenMaya.MTypeId(0xd802b2)
    OPERATOR = operator.truediv

    # def __init__(self):
    #     OpenMaya.MPxNode.__init__(self)
    #     super().__init__()

class RRM3Vec3AddNode(RRM3Vec3Node):
    """
    A vector3 divide node
    """
    TYPE_NAME = 'rrm3Vec3Add'
    TYPE_ID = OpenMaya.MTypeId(0xd802b3)
    OPERATOR = operator.add


class RRM3Vec3SubtractNode(RRM3Vec3Node):
    """
    A vector3 divide node
    """
    TYPE_NAME = 'rrm3Vec3Subtract'
    TYPE_ID = OpenMaya.MTypeId(0xd802b4)
    OPERATOR = operator.sub




