import maya.cmds as cmds
from rrm3.Contents.scripts import add_set_attrs
from rrm3.Contents.scripts.build_nodes import math_nodes
from rrm3.Contents.scripts.control_rig import rig_data
from rrm3.Contents.scripts import debug_info

# import importlib
# plugins_2022 = importlib.import_module()
# from rrm3.Contents.plug-ins.2022 import rapidrig_modular_v3
import operator

'''
all Maya nodes should be built through this so that they are all connected to the rig
'''


def create_jnt(jnt_name, parent='', radius=1.0, outliner_color=16):
    # print('creating joint', jnt_name)
    if parent:
        cmds.select(parent)
    else:
        cmds.select(clear=True)
    jnt = cmds.joint(name=jnt_name)
    # print(f'{jnt_name=}, {radius=}')
    cmds.setAttr(f'{jnt}.radius', radius)
    add_set_attrs.set_outliner_color(node=jnt, outliner_color=outliner_color)

    return jnt


class MayaNode(object):
    # constant names
    JNT_SUFFIX = rig_data.RigData.JNT_SUFFIX
    CTRL_SUFFIX = rig_data.RigData.CTRL_SUFFIX
    GRP_SUFFIX = rig_data.RigData.GRP_SUFFIX
    LCTR_SUFFIX = rig_data.RigData.LCTR_SUFFIX
    IK_SUFFIX = rig_data.RigData.IK_SUFFIX
    FK_SUFFIX = rig_data.RigData.FK_SUFFIX
    IK_HANDLE_SUFFIX = rig_data.RigData.IK_HANDLE_SUFFIX
    NULL_DRIVEN_SUFFIX = rig_data.RigData.NULL_DRIVEN_SUFFIX

    LEFT_PREFIX = rig_data.RigData.left_prefix
    RIGHT_PREFIX = rig_data.RigData.right_prefix

    def __init__(self, rig_data_obj=None):
        self.rig_data_obj = rig_data_obj
        self.rig_name = self.get_rig_name()
        self.debug_vis = rig_data_obj.debug_vis
        self.top_node = self.get_top_node()

        self.message_top_node_attr = rig_data_obj.MESSAGE_TOP_NODE_ATTR
        self.message_attr = rig_data_obj.MESSAGE_ATTR
        self.message_skin_jnt_attr = rig_data_obj.MESSAGE_SKIN_JNT_ATTR
        self.message_skin_sh_jnt_attr = rig_data_obj.MESSAGE_SKIN_SH_JNT_ATTR
        self.message_ctrl_output_attr = rig_data_obj.MESSAGE_CTRL_OUTPUT_ATTR
        self.message_ctrl_input_attr = rig_data_obj.MESSAGE_CTRL_INPUT_ATTR
        self.message_grp_output_attr = rig_data_obj.MESSAGE_GRP_OUTPUT_ATTR
        self.message_grp_input_attr = rig_data_obj.MESSAGE_GRP_INPUT_ATTR
        self.sh_unreal_naming = rig_data_obj.sh_unreal_naming
        self.sh_unreal_orient = rig_data_obj.sh_unreal_orient
        self.generate_rapid_mesh = rig_data_obj.generate_rapid_mesh

    def get_rig_name(self):
        # print(f'get_rig_name {self.rig_data_obj=}')
        if self.rig_data_obj.rig_name:
            return self.rig_data_obj.rig_name
        else:
            return ''

    def get_top_node(self):
        # print(f'getting top node {self.rig_name}')
        if self.rig_name:
            return self.rig_name
        else:
            return 'controlRig'

    def maya_grp(self, name, child='', parent='', suffix=GRP_SUFFIX, match_target='', match_translate=True,
                 match_rotate=True, match_scale=True, visible_in_outliner=True, is_grp=True, is_top_grp=False,
                 inherit_transform=True, outliner_color=16):
        """

        @param str name: group name
        @param str, list child: parents child or children to group
        @param str parent: parent of group
        @param str suffix: group suffix
        @param str match_target: target to match transforms to
        @param bool match_translate: match translate
        @param bool match_rotate: match rotate
        @param bool match_scale: match scale
        @param bool visible_in_outliner: if true, hides in outliner and sets visibility to False
        @param bool is_grp: connects to the group attr of the rig, and will lock all channels
        @param is_top_grp: do not connect top group to itself
        @param inherit_transform: set whether to inherit transforms
        @param outliner_color:
        @return:
        """
        grp = cmds.group(name=f'{name}{suffix}', empty=True)

        if not is_top_grp:
            self.connect_to_top_node(grp)
        if parent:
            cmds.parent(grp, parent)
        if match_target:
            cmds.matchTransform(grp, match_target, position=match_translate, rotation=match_rotate, scale=match_scale)

        if child:
            cmds.parent(child, grp)

        if is_grp and not is_top_grp:
            add_set_attrs.add_and_set(grp, self.message_grp_input_attr, attr_type='message')
            # print(f'{self.rig_data_obj.top_grp=}')
            add_set_attrs.connect_attr(self.top_node, self.message_grp_output_attr,
                                       grp, self.message_grp_input_attr)

        cmds.setAttr(f'{grp}.inheritsTransform', inherit_transform)

        if not visible_in_outliner:
            cmds.setAttr(f'{grp}.hiddenInOutliner', True)
            cmds.setAttr(f'{grp}.visibility', False)

        add_set_attrs.set_outliner_color(node=grp, outliner_color=outliner_color)

        return grp

    def utility(self, node_name, node_type, replace_existing=True, outliner_color=16):
        utility_node_name = f'{node_name}_{node_type}'
        if cmds.objExists(utility_node_name) and replace_existing:
            # cmds.warning('deleted existing node {node}')
            self.delete_node(utility_node_name)
        if not cmds.objExists(utility_node_name):
            rrm_nodes = ['rrm3FloatMult', 'rrm3FloatDiv', 'rrm3FloatAdd', 'rrm3FloatSubtract',
                         'rrm3Vec3Mult', 'rrm3Vec3Div', 'rrm3Vec3Add', 'rrm3Vec3Subtract',
                         ]
            if node_type in rrm_nodes:
                # print(f'{node_type=}')
                # print(f'{utility_node_name=}')
                utility_node_name = cmds.createNode(node_type, name=utility_node_name)
            else:
                utility_node_name = cmds.shadingNode(node_type, name=utility_node_name, asUtility=True)
            self.connect_to_top_node(utility_node_name)

        # if cmds.attributeQuery('isHistoricallyInteresting', node=utility_node_name, exists=True):
        cmds.setAttr(f'{utility_node_name}.isHistoricallyInteresting', False)

        return utility_node_name

    def locator(self, lctr_name, parent_node='', inherits_transforms=True, suffix=LCTR_SUFFIX, match_node='',
                outliner_color=16):
        lctr = cmds.spaceLocator(name=f'{lctr_name}{suffix}')[0]
        cmds.setAttr(f'{lctr}.visibility', self.debug_vis)
        cmds.setAttr(f'{lctr}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))
        if parent_node:
            cmds.parent(lctr, parent_node)
        if not inherits_transforms:
            cmds.setAttr(f'{lctr}.inheritsTransform', False)
        if match_node:
            cmds.matchTransform(lctr, match_node)

        self.connect_to_top_node(lctr)
        add_set_attrs.set_outliner_color(node=lctr, outliner_color=outliner_color)

        return lctr

    def skinning_jnt(self, jnt_name, parent: str = '', radius=1.0, mod_grp='', sh_jnt=False, outliner_color=16):
        """ add special message connection for selecting skinning joints """
        jnt = self.create_base_jnt(jnt_name=jnt_name, parent=parent, radius=radius, mod_grp=mod_grp,
                                   outliner_color=outliner_color)
        attr_name = self.message_skin_sh_jnt_attr if sh_jnt else self.message_skin_jnt_attr
        add_set_attrs.add_and_set(node=jnt, attr_name=attr_name, attr_type='message')
        add_set_attrs.connect_attr(self.top_node, attr_name, jnt, attr_name)

        # add label to joint
        cmds.setAttr(f'{jnt}.type', 18)
        if not self.sh_unreal_naming:  # standard naming check
            if jnt_name.startswith(self.rig_data_obj.LEFT_PREFIX):
                cmds.setAttr(f'{jnt}.side', 1)
                cmds.setAttr(f'{jnt}.otherType', jnt_name[len(self.rig_data_obj.LEFT_PREFIX):], type='string')
            elif jnt_name.startswith(self.rig_data_obj.RIGHT_PREFIX):
                cmds.setAttr(f'{jnt}.side', 2)
                cmds.setAttr(f'{jnt}.otherType', jnt_name[len(self.rig_data_obj.RIGHT_PREFIX):], type='string')
            else:
                cmds.setAttr(f'{jnt}.otherType', jnt_name, type='string')

        else:
            if jnt_name.endswith('_l'):
                cmds.setAttr(f'{jnt}.side', 1)
                cmds.setAttr(f'{jnt}.otherType', jnt_name[:-2], type='string')
            elif jnt_name.endswith('_r'):
                cmds.setAttr(f'{jnt}.side', 2)
                cmds.setAttr(f'{jnt}.otherType', jnt_name[:-2], type='string')

            else:
                cmds.setAttr(f'{jnt}.otherType', jnt_name, type='string')

        return jnt

    def non_skinning_jnt(self, jnt_name, parent: str = '', radius=1.0, mod_grp='', outliner_color=16):
        """ create a joint used for deforming the control rig """
        jnt = self.create_base_jnt(jnt_name, parent, radius, mod_grp=mod_grp, outliner_color=outliner_color)
        cmds.setAttr(f'{jnt}.visibility', self.debug_vis)
        cmds.setAttr(f'{jnt}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))

        return jnt

    def create_base_jnt(self, jnt_name, parent, radius=1.0, mod_grp='', outliner_color=16):
        jnt = create_jnt(jnt_name=jnt_name, parent=parent, radius=radius, outliner_color=outliner_color)
        self.connect_to_top_node(jnt)
        if mod_grp:
            self.connect_to_module(jnt, mod_grp=mod_grp)
        return jnt

    def connect_to_top_node(self, node):
        add_set_attrs.add_and_set(node, self.message_attr, attr_type='message')
        add_set_attrs.connect_attr(self.top_node, self.message_top_node_attr, node, self.message_attr)

    @staticmethod
    def connect_to_module(node, attr_name='moduleJnts', mod_grp=''):
        if mod_grp:
            add_set_attrs.add_and_set(node, 'moduleGrp', attr_type='message')
            add_set_attrs.connect_attr(source_obj=mod_grp, source_attr=attr_name,
                                       target_objs=node, target_attr='moduleGrp')

    def connect_node_to_attr(self, node, attr):
        if not cmds.attributeQuery(attr, node=node, exists=True):
            # cmds.addAttr(self.top_node, longName=attr, attributeType='message')
            add_set_attrs.add_and_set(self.top_node, attr_name=attr, attr_type='message')

        if not cmds.attributeQuery(attr, node=node, exists=True):
            add_set_attrs.add_and_set(node, attr_name=attr, attr_type='message')

        add_set_attrs.connect_attr(self.top_node, attr, node, attr)

    def ik_handle(self, name, start_jnt, end_effector, solver, parent='', pole_vector_tgt='', outliner_color=16):
        ik_handle, effector = cmds.ikHandle(name=f'{name}{self.IK_HANDLE_SUFFIX}',
                                            startJoint=start_jnt,
                                            endEffector=end_effector,
                                            solver=solver)

        self.maya_grp(ik_handle, child=ik_handle, match_target=end_effector, match_scale=False, parent=parent,
                      visible_in_outliner=self.rig_data_obj.debug_vis)
        if solver == 'ikRPsolver' and pole_vector_tgt:
            # cmds.poleVectorConstraint(pole_vector_tgt, ik_handle)
            PoleVectorConstraint(rig_data_obj=self.rig_data_obj, targets=pole_vector_tgt, constrained_node=ik_handle)

        cmds.setAttr(f'{ik_handle}.visibility', self.debug_vis)
        cmds.setAttr(f'{ik_handle}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))
        cmds.setAttr(f'{effector}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))

        self.connect_to_top_node(ik_handle)
        self.connect_to_top_node(effector)
        add_set_attrs.set_outliner_color(node=ik_handle, outliner_color=outliner_color)
        add_set_attrs.set_outliner_color(node=effector, outliner_color=outliner_color)

        return ik_handle

    def spline_ik_handle(self, name, start_jnt, end_effector, parent_node=None, ik_curve=None, outliner_color=16):
        if ik_curve:
            ik_handle, effector, = cmds.ikHandle(name=f'{name}Spline{self.IK_HANDLE_SUFFIX}',
                                                 startJoint=start_jnt,
                                                 endEffector=end_effector,
                                                 solver='ikSplineSolver',
                                                 createCurve=False,
                                                 curve=ik_curve,
                                                 parentCurve=False)
        else:
            ik_handle, effector, ik_curve = cmds.ikHandle(name=f'{name}Spline{self.IK_HANDLE_SUFFIX}',
                                                          startJoint=start_jnt,
                                                          endEffector=end_effector,
                                                          solver='ikSplineSolver')

        effector = cmds.rename(effector, f'{name}Spline{self.IK_HANDLE_SUFFIX}Effector')

        cmds.setAttr(f'{ik_handle}.visibility', self.debug_vis)
        cmds.setAttr(f'{ik_curve}.visibility', self.debug_vis)
        cmds.setAttr(f'{ik_handle}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))
        cmds.setAttr(f'{effector}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))
        cmds.setAttr(f'{ik_curve}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))
        add_set_attrs.set_outliner_color(node=ik_handle, outliner_color=outliner_color)
        add_set_attrs.set_outliner_color(node=ik_curve, outliner_color=outliner_color)
        add_set_attrs.set_outliner_color(node=effector, outliner_color=outliner_color)

        cmds.setAttr(f'{ik_handle}.isHistoricallyInteresting', False)
        cmds.setAttr(f'{ik_curve}.isHistoricallyInteresting', False)
        cmds.setAttr(f'{effector}.isHistoricallyInteresting', False)

        self.connect_to_top_node(ik_handle)
        self.connect_to_top_node(effector)
        self.connect_to_top_node(ik_curve)

        if parent_node:
            cmds.parent(ik_handle, parent_node)

        return ik_handle, effector, ik_curve

    def arc_len(self, name, curve):
        arc_len = cmds.arclen(curve, constructionHistory=True)
        arc_len = cmds.rename(arc_len, f'{name}_ArcLen')
        self.connect_to_top_node(arc_len)
        cmds.setAttr(f'{arc_len}.isHistoricallyInteresting', False)

        return arc_len

    def cluster(self, name, points, parent='', match_node='', match_translate=True, match_rotate=True,
                match_scale=True, outliner_color=16):
        cluster, cluster_handle = cmds.cluster(points, name=f'{name}_Clstr')

        if parent:
            cmds.parent(cluster_handle, parent)
        if match_node:
            cmds.matchTransform(cluster_handle, match_node, position=match_translate, rotation=match_rotate,
                                scale=match_scale)
        cmds.setAttr(f'{cluster_handle}.visibility', self.rig_data_obj.debug_vis)
        cmds.setAttr(f'{cluster_handle}.hiddenInOutliner', operator.not_(self.rig_data_obj.debug_vis))
        # add_set_attrs.set_outliner_color(node=cluster, outliner_color=outliner_color)
        add_set_attrs.set_outliner_color(node=cluster_handle, outliner_color=outliner_color)
        cmds.setAttr(f'{cluster}.isHistoricallyInteresting', False)
        cmds.setAttr(f'{cluster_handle}.isHistoricallyInteresting', False)

        return [cluster, cluster_handle]

    def controller(self, child_ctrl, parent_ctrl=None):
        if parent_ctrl:
            cmds.controller(child_ctrl, parent_ctrl, parent=True)
        else:
            cmds.controller(child_ctrl, parent=True)
        controller = cmds.listConnections(child_ctrl, type='controller')[0]
        cmds.setAttr(f'{controller}.isHistoricallyInteresting', False)

        self.connect_to_top_node(controller)

    def display_layer(self, layer_nodes, layer_name, connection_attr=''):
        """
        creates a display layer
        @param list, str layer_nodes: a list of nodes to add to the layer
        @param layer_name: the name of the display layer
        @param connection_attr: attribute to connect to the top node
        """
        if not isinstance(layer_nodes, list):
            layer_nodes = [layer_nodes]
        cmds.createDisplayLayer(layer_nodes, name=layer_name, noRecurse=True)

        self.connect_to_top_node(layer_name)
        if connection_attr:
            for layer_node in layer_nodes:
                self.connect_node_to_attr(layer_node, connection_attr)

    @classmethod
    def delete_node(cls, node_name):
        cmds.lockNode(node_name, lock=False)
        child_nodes = cmds.listRelatives(node_name)
        if child_nodes:
            for child_node in child_nodes:
                if cmds.objExists(child_node):
                    child_nodes.lockNode(lock=False)
        cmds.delete(node_name)


class BaseConstraint(MayaNode):
    def __init__(self, rig_data_obj, targets, constrained_node):
        """
        @param list, str targets:
        @param str constrained_node:
        """
        super().__init__(rig_data_obj)

        self.constraint_grp = f'{self.rig_data_obj.top_grp}_Constraint{self.GRP_SUFFIX}'

        self.constraint: str = ''
        self.weight_target_dict: str = ''

        self.targets = self.get_targets(targets=targets)
        self.constrained_node = constrained_node

    @staticmethod
    def get_targets(targets):
        """
        returns targets as a list, regardless of whether it is a string or list
        @param targets:
        @return:
        """
        targets = targets if isinstance(targets, list) else [targets]
        return targets

    def connect_to_rig(self):
        self.create_constraint_grp()

        add_set_attrs.add_and_set(self.constraint, attr_name=self.rig_data_obj.MESSAGE_CONSTRAINT_INPUT_ATTR,
                                  attr_type='message')

        add_set_attrs.connect_attr(self.rig_data_obj.top_grp, self.rig_data_obj.MESSAGE_CONSTRAINT_OUTPUT_ATTR,
                                   self.constraint, self.rig_data_obj.MESSAGE_CONSTRAINT_INPUT_ATTR)

        if type(self) in [ParentConstraint, OrientConstraint]:
            cmds.setAttr(f'{self.constraint}.interpType', self.interp_type)

    def create_constraint_grp(self):
        """
        put constraints into a group if not in debug mode
        @return:
        """
        if not debug_info.release_mode:
            return

        if not cmds.objExists(self.constraint_grp):
            self.maya_grp(name=self.constraint_grp,
                          parent=self.rig_data_obj.top_grp,
                          child=self.constraint,
                          visible_in_outliner=self.rig_data_obj.debug_vis,
                          suffix='')
        else:
            cmds.parent(self.constraint, self.constraint_grp)

    def get_weights_and_targets(self):
        """
        gets all the weight attr's of the constraints
        @return dict: key as the weight #, target as the attribute name
        """
        weights = cmds.listAttr(self.constraint, userDefined=True)
        weights_target_dict = {}
        for i, weight in enumerate(weights):
            weights_target_dict[i] = weight

        return weights_target_dict


class Constraint(BaseConstraint):
    def __init__(self, rig_data_obj, targets, constrained_node, maintain_offset=False, interp_type=2):
        super().__init__(rig_data_obj, targets, constrained_node)

        self.maintain_offset = maintain_offset
        self.interp_type = interp_type


class ParentConstraint(Constraint):
    def __init__(self, rig_data_obj, targets, constrained_node,
                 maintain_offset=False,
                 interp_type=2,
                 skip_translate=[], skip_rotate=[]):
        """

        @param rig_data_obj:
        @param list, str targets:
        @param str constrained_node:
        @param bool maintain_offset:
        @param int interp_type:
        @param list skip_translate:
        @param list skip_rotate:
        """
        super().__init__(rig_data_obj, targets, constrained_node, maintain_offset, interp_type)

        self.skip_translate = skip_translate
        self.skip_rotate = skip_rotate

        self.constraint = self.make_constraint()
        self.weight_target_dict = self.get_weights_and_targets()
        self.connect_to_rig()

    def make_constraint(self):
        constraint = cmds.parentConstraint(self.targets, self.constrained_node,
                                           maintainOffset=self.maintain_offset,
                                           skipRotate=self.skip_rotate, skipTranslate=self.skip_translate)[0]

        return constraint


class PoleVectorConstraint(BaseConstraint):
    def __init__(self, rig_data_obj, targets, constrained_node):
        super().__init__(rig_data_obj, targets, constrained_node)
        self.targets = targets
        self.constrained_node = constrained_node  # this is an ik handle

        self.constraint = self.make_constraint()
        self.weight_target_dict = self.get_weights_and_targets()
        self.connect_to_rig()

    def make_constraint(self):
        constraint = cmds.poleVectorConstraint(self.targets, self.constrained_node)[0]
        return constraint


class TransformConstraint(Constraint):
    def __init__(self, rig_data_obj, targets, constrained_node, maintain_offset=False, interp_type=2, skip=[]):
        super().__init__(rig_data_obj, targets, constrained_node, maintain_offset, interp_type)

        self.skip = skip


class PointConstraint(TransformConstraint):  # point constraint
    def __init__(self, rig_data_obj, targets, constrained_node, maintain_offset=False, interp_type=None, skip=[]):
        super().__init__(rig_data_obj, targets, constrained_node, maintain_offset, interp_type, skip)

        self.constraint = self.make_constraint()
        self.weight_target_dict = self.get_weights_and_targets()
        self.connect_to_rig()

    def make_constraint(self):
        constraint = cmds.pointConstraint(self.targets, self.constrained_node,
                                          skip=self.skip,
                                          maintainOffset=self.maintain_offset)[0]
        return constraint


class OrientConstraint(TransformConstraint):  # orient consraint
    def __init__(self, rig_data_obj, targets, constrained_node, maintain_offset=False, interp_type=2, skip=[]):
        super().__init__(rig_data_obj, targets, constrained_node, maintain_offset, interp_type, skip)
        self.constraint = self.make_constraint()
        self.weight_target_dict = self.get_weights_and_targets()
        self.connect_to_rig()

    def make_constraint(self):
        constraint = cmds.orientConstraint(self.targets, self.constrained_node,
                                           skip=self.skip,
                                           maintainOffset=self.maintain_offset)[0]
        return constraint


class ScaleConstraint(TransformConstraint):  # scale constraint
    def __init__(self, rig_data_obj, targets, constrained_node, maintain_offset=False, interp_type=None, skip=[]):
        super().__init__(rig_data_obj, targets, constrained_node, maintain_offset, interp_type, skip)

        self.constraint = self.make_constraint()
        self.weight_target_dict = self.get_weights_and_targets()
        self.connect_to_rig()

    def make_constraint(self):
        constraint = cmds.scaleConstraint(self.targets, self.constrained_node,
                                          skip=self.skip,
                                          maintainOffset=self.maintain_offset)[0]
        return constraint


class AimConstraint(TransformConstraint):
    OBJECT_UP_TYPES = ['object', 'objectrotation']

    def __init__(self, rig_data_obj, targets, constrained_node, maintain_offset=False, interp_type=None, skip=[],
                 aim_vector=[1, 0, 0], up_vector=[0, 1, 0],
                 world_up_type='vector', world_up_object='', world_up_vector=[0, 1, 0]):
        super().__init__(rig_data_obj, targets, constrained_node, maintain_offset, interp_type, skip)

        self.aim_vector = aim_vector
        self.up_vector = up_vector
        self.world_up_vector = world_up_vector
        self.world_up_object = world_up_object
        self.world_up_type = world_up_type

        self.constraint = self.make_constraint()
        self.weight_target_dict = self.get_weights_and_targets()
        self.connect_to_rig()

    def make_constraint(self):

        if self.world_up_type in self.OBJECT_UP_TYPES:
            constraint = cmds.aimConstraint(self.targets, self.constrained_node,
                                            aimVector=self.aim_vector,
                                            upVector=self.up_vector,
                                            worldUpType=self.world_up_type,
                                            worldUpObject=self.world_up_object,
                                            worldUpVector=self.world_up_vector,
                                            skip=self.skip,
                                            maintainOffset=self.maintain_offset)[0]
        else:
            constraint = cmds.aimConstraint(self.targets, self.constrained_node,
                                            aimVector=self.aim_vector,
                                            upVector=self.up_vector,
                                            worldUpType=self.world_up_type,
                                            worldUpVector=self.world_up_vector,
                                            skip=self.skip,
                                            maintainOffset=self.maintain_offset)[0]
        return constraint


def create_sdk(driver, driver_attr, driven, driven_attrs=None):
    if not driven_attrs:
        driven_attrs = ['tx', 'ty', 'tz', 'rx', 'ry', 'rz', 'sx', 'sy', 'sz']
    channel_val = {}
    for driven_attr in driven_attrs:
        locked_attr = cmds.getAttr(f'{driven}.{driven_attr}', lock=True)
        if locked_attr:
            cmds.setAttr(f'{driven}.{driven_attr}', lock=False)

        cmds.setDrivenKeyframe(driven, attribute=f'{driven_attr}',
                               currentDriver=f'{driver}.{driver_attr}',
                               inTangentType='clamped', outTangentType='clamped')
        channel_val[f'{driven_attr}'] = cmds.getAttr(f'{driven}.{driven_attr}')
        if locked_attr:
            cmds.setAttr(f'{driven}.{driven_attr}', lock=True)

    return channel_val


def create_mult_double(name, input1_plug=None, input2_plug=None, output_plug=None):
    # rapidrig_modular_v3.RRM3FloatMultNode()
    math_nodes.RRM3FloatMultNode()
