# CopyRight Dustin Nelson 2010-2025. Please do not distribute
# https://www.rapidrig.com
# Plug-in for Modular Character Rigging.

"""
Rapid Rig Modular (RRM) is for production and personal use only.
RRM rigs, in whole or in part, or any of the RRM code is not to be distributed outside of a production.
RRM is not to be used for rigging models intended for sale, trade or any other distribution.
"""

# version #, eg 2025, is set up through the PackageContents.xml, this file needs to be updated with each Maya release

from importlib import reload
import sys

import inspect

import maya.api.OpenMaya as OpenMaya


dir_path = inspect.getfile(lambda: None)
rrm3_path = dir_path.rsplit('rrm3', 1)[0]

if rrm3_path not in sys.path:
    sys.path.append(rrm3_path)

from rrm3.Contents.scripts.gui import main_gui_create_qt, maya_shelf
from rrm3.Contents.scripts import global_variables
from rrm3.Contents.scripts.selector.gui import selector_gui
from rrm3.Contents.scripts.build_nodes import math_nodes
from rrm3.Contents.scripts.control_rig.transforms import custom_transforms

from rrm3.Contents.scripts.marking_menus import mm_animation

maya_useNewAPI = True

version_number = global_variables.get_version_number()
version_int = int(version_number.split('.')[0])
rrm_prefix = f'RRM{version_int}'


class RapidRigModular3Cmd(OpenMaya.MPxCommand):
    TYPE_NAME = 'rapidRigModular3'

    def __init__(self):
        OpenMaya.MPxCommand.__init__(self)

    @staticmethod
    def cmdCreator():
        return RapidRigModular3Cmd()

    def doIt(self, args):
        all_rrm3_modules = sys.modules.values()
        for rrm3_module in list(all_rrm3_modules):
            rrm3_mod_file_path = str(rrm3_module)
            if 'rrm3' in rrm3_mod_file_path and rrm3_mod_file_path.endswith(".py'>"):
                reload(rrm3_module)

        # load ui
        main_gui_create_qt.create_main_gui(version_number)


class RapidRigModular3SelectorCmd(OpenMaya.MPxCommand):
    version_number = global_variables.get_version_number()
    version_int = int(version_number.split('.')[0])
    rrm_prefix = f'RRM{version_int}'

    TYPE_NAME = 'rapidRigModular3Selector'

    def __init__(self):
        OpenMaya.MPxCommand.__init__(self)

    @staticmethod
    def cmdCreator():
        return RapidRigModular3SelectorCmd()

    def doIt(self, args):
        # reload all the plugins
        all_rrm3_modules = sys.modules.values()
        for rrm3_module in list(all_rrm3_modules):
            rrm3_mod_file_path = str(rrm3_module)
            if 'rrm3' in rrm3_mod_file_path and rrm3_mod_file_path.endswith(".py'>"):
                reload(rrm3_module)

        # load ui
        selector_gui.create_selector_gui(version_number)


class RapidRigModular3MarkingMenuCmd(OpenMaya.MPxCommand):
    TYPE_NAME = 'rapidRigModular3MarkingMenu'

    def __init__(self):
        OpenMaya.MPxCommand.__init__(self)
        # self.enable = enable
        # create mel file for the marking menus

    @staticmethod
    def cmdCreator():
        return RapidRigModular3MarkingMenuCmd()

    def doIt(self, args):
        # reload all the plugins
        all_rrm3_modules = sys.modules.values()
        for rrm3_module in list(all_rrm3_modules):
            rrm3_mod_file_path = str(rrm3_module)
            if 'rrm3' in rrm3_mod_file_path and rrm3_mod_file_path.endswith(".py'>"):
                reload(rrm3_module)

        mm_animation.launch_settings_ui()


rrm3_math_nodes = [math_nodes.RRM3FloatMultNode, math_nodes.RRM3FloatDivNode,
                   math_nodes.RRM3FloatAddNode, math_nodes.RRM3FloatSubtractNode,
                   math_nodes.RRM3Vec3MultNode, math_nodes.RRM3Vec3DivNode,
                   math_nodes.RRM3Vec3AddNode, math_nodes.RRM3Vec3SubtractNode,
                   ]

transform_nodes = [custom_transforms.RRM3Control, custom_transforms.RRM3ControlGroup]
transform_nodes = None  # putting this here because they will not load


def initializePlugin(m_object_plugin):
    """ initialize plugin """
    fn_plugin = OpenMaya.MFnPlugin(m_object_plugin, 'Rapid Rig', '3.0.1', 'Any')

    try:
        fn_plugin.registerCommand(RapidRigModular3Cmd.TYPE_NAME,
                                  RapidRigModular3Cmd.cmdCreator)
        fn_plugin.registerCommand(RapidRigModular3SelectorCmd.TYPE_NAME,
                                  RapidRigModular3SelectorCmd.cmdCreator)
        fn_plugin.registerCommand(RapidRigModular3MarkingMenuCmd.TYPE_NAME,
                                  RapidRigModular3MarkingMenuCmd.cmdCreator)
        # import math nodes
        for rrm3_math_node in rrm3_math_nodes:
            fn_plugin.registerNode(rrm3_math_node.TYPE_NAME,
                                   rrm3_math_node.TYPE_ID,
                                   rrm3_math_node.creator,
                                   rrm3_math_node.initialize,
                                   OpenMaya.MPxNode.kDependNode)

        if transform_nodes:
            for transform_node in transform_nodes:
                fn_plugin_mpx.registerTransform(
                    transform_node.TYPE_NAME,
                    transform_node.TYPE_ID,
                    transform_node.creator,
                    transform_node.initialize,
                    transform_node.matrix_creator,
                    custom_transforms.RRM3TransformMatrix.initialize,
                    custom_transforms.RRM3TransformMatrix.TYPE_ID,
                    # OpenMayaMPx.MPxTransform.kTransform
                )
        # OpenMayaMpx.MFnPlugin.registerTransform(kTransformNodeName,kTransformNodeId,transformCreator,transformNodeInitializer, matrixCreator,kTransformMatrixNodeId)

        print('Rapid Rig Modular V3 successfully loaded.')
    except RuntimeError as err:
        print(f'failed to initialize plugin: {RapidRigModular3Cmd.TYPE_NAME}')
        print(f'failed to initialize plugin: {RapidRigModular3SelectorCmd.TYPE_NAME}')
        print(f'failed to initialize plugin: {RapidRigModular3MarkingMenuCmd.TYPE_NAME}')
        # math nodes
        for rrm3_math_node in rrm3_math_nodes:
            print(f'failed to initialize plugin: {rrm3_math_node.TYPE_NAME}')
        if transform_nodes:
            for transform_node in transform_nodes:
                print(f'failed to initialize plugin: {transform_node.TYPE_NAME}')
        raise RuntimeError("Failed to register nodes: {}".format(err))


def uninitializePlugin(plugin):
    """ un-initialize plugin """
    fn_plugin = OpenMaya.MFnPlugin(plugin)

    try:
        fn_plugin.deregisterCommand(RapidRigModular3Cmd.TYPE_NAME)
        fn_plugin.deregisterCommand(RapidRigModular3SelectorCmd.TYPE_NAME)
        fn_plugin.deregisterCommand(RapidRigModular3MarkingMenuCmd.TYPE_NAME)
        for rrm3_math_node in rrm3_math_nodes:
            fn_plugin.deregisterNode(rrm3_math_node.TYPE_ID)

        if transform_nodes:
            for transform_node in transform_nodes:
                fn_plugin.deregisterNode(transform_node.TYPE_ID)

        print('Rapid Rig Modular V3 successfully un-initialize plugin.')

    except RuntimeError as err:
        print(f'failed to un-initialize plugin: {RapidRigModular3Cmd.TYPE_NAME}')
        print(f'failed to un-initialize plugin: {RapidRigModular3SelectorCmd.TYPE_NAME}')
        print(f'failed to un-initialize plugin: {RapidRigModular3MarkingMenuCmd.TYPE_NAME}')
        # print(f'failed to un-initialize plugin: {math_nodes.RRM3FloatMultNode.TYPE_NAME}')
        for rrm3_math_node in rrm3_math_nodes:
            print(f'failed to un-initialize plugin: {rrm3_math_node.TYPE_NAME}')
        if transform_nodes:
            for transform_node in transform_nodes:
                print(f'failed to un-initialize plugin: {transform_node.TYPE_NAME}')
        raise RuntimeError("Failed to deregister nodes: {}".format(err))


def rrm3():
    main_gui_create_qt.create_main_gui(version_number)


maya_shelf.create_shelf()
